// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A4__F49

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_1___gfx11xx)xyzw" };
#else
{ R"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_1___gfx11xx)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_64_0_F_F_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1151)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xcff7bf6au, 0x74e3d0f4u, 0, 46 }, // cff7bf6a74e3d0f4 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_1___gfx11xx__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1151
    { 0xba834606u, 0x3a35083au, 0, 63 }, // ba8346063a35083a = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_1___gfx11xx__P__16_16__CO__wave2_warp2_stg1--Arch_gfx1151
    { 0xf5dfb521u, 0xb52d3c04u, 40, 46 }, // f5dfb521b52d3c04 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_1___gfx11xx__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1151
    { 0x20a2d1e4u, 0x48f2898du, 40, 63 }, // 20a2d1e448f2898d = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_1___gfx11xx__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1151
    { 0xffcf720bu, 0x5b127b1cu, 40, 80 }, // ffcf720b5b127b1c = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_1___gfx11xx__P__32_16__CO__wave2_warp4_stg1--Arch_gfx1151
    { 0x67aa3d18u, 0x9c7ddc38u, 40, 6 }, // 67aa3d189c7ddc38 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_1___gfx11xx__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1151
    { 0xa441148fu, 0x5094b3d8u, 40, 23 }, // a441148f5094b3d8 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_1___gfx11xx__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1151
    { 0x6d4dac8bu, 0xfa0d05f2u, 97, 138 }, // 6d4dac8bfa0d05f2 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_1___gfx11xx__P__32_32__CO__wave1_warp4_stg1--Arch_gfx1151
    { 0x59b638e4u, 0x631dd646u, 97, 80 }, // 59b638e4631dd646 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_1___gfx11xx__P__32_32__CO__wave2_warp4_stg1--Arch_gfx1151
    { 0x649f50b6u, 0xb91111a2u, 120, 138 }, // 649f50b6b91111a2 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_1___gfx11xx__P__64_16__CO__wave1_warp4_stg1--Arch_gfx1151
    { 0x9f942018u, 0x56957618u, 120, 80 }, // 9f94201856957618 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_1___gfx11xx__P__64_16__CO__wave2_warp4_stg1--Arch_gfx1151
    { 0x269b289au, 0xf5e3616au, 120, 155 }, // 269b289af5e3616a = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_1___gfx11xx__P__64_16__CO__wave3_warp4_stg1--Arch_gfx1151
    { 0x3dea302eu, 0x6a93b3b9u, 120, 103 }, // 3dea302e6a93b3b9 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_64_0_F_F_1___gfx11xx__P__64_16__CO__wave4_warp4_stg1--Arch_gfx1151
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1151_mod0
{{ 4, 6, 0, 0, 0, 0, 0, 0, 0, 0},
 {10,10, 9, 3, 6, 6, 6, 2, 6, 2},
 { 8, 5, 6,12,12, 6, 6, 2, 3, 2},
 { 7, 0, 3,12,10, 6, 1, 5, 5, 3},
 { 7, 0, 2,10,10, 6, 1, 3,12,12},
 { 0, 5, 3,10,12, 5, 5, 3, 9, 8},
 { 2, 2, 3,10,10,10, 6, 2, 8, 7},
 { 5, 5, 5, 9,11,10,10, 6, 6, 5},
 { 2, 3, 3,11,12,12, 1, 1, 5, 2},
 { 2, 3,10,10,11, 6, 1, 6, 3, 2}}
// End of GPU gfx1151_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A4__F49 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 1;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 1;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = 0
// BLOCK_DMODEL = 64
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = False
// PADDED_HEAD = False
// BIAS_TYPE = 1

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx1100_mod0') AND inputs$Q_dtype = 'torch.float16' AND inputs$BLOCK_DMODEL = 64 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = False AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 1

// vim: set fileencoding=utf-8

