{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- A @GdkPixbufModule@ contains the necessary functions to load and save
-- images in a certain file format.
-- 
-- If @GdkPixbuf@ has been compiled with @GModule@ support, it can be extended
-- by modules which can load (and perhaps also save) new image and animation
-- formats.
-- 
-- == Implementing modules
-- 
-- The @GdkPixbuf@ interfaces needed for implementing modules are contained in
-- @gdk-pixbuf-io.h@ (and @gdk-pixbuf-animation.h@ if the module supports
-- animations). They are not covered by the same stability guarantees as the
-- regular GdkPixbuf API. To underline this fact, they are protected by the
-- @GDK_PIXBUF_ENABLE_BACKEND@ pre-processor symbol.
-- 
-- Each loadable module must contain a @GdkPixbufModuleFillVtableFunc@ function
-- named @fill_vtable@, which will get called when the module
-- is loaded and must set the function pointers of the @GdkPixbufModule@.
-- 
-- In order to make format-checking work before actually loading the modules
-- (which may require calling @dlopen@ to load image libraries), modules export
-- their signatures (and other information) via the @fill_info@ function. An
-- external utility, @gdk-pixbuf-query-loaders@, uses this to create a text
-- file containing a list of all available loaders and  their signatures.
-- This file is then read at runtime by @GdkPixbuf@ to obtain the list of
-- available loaders and their signatures.
-- 
-- Modules may only implement a subset of the functionality available via
-- @GdkPixbufModule@. If a particular functionality is not implemented, the
-- @fill_vtable@ function will simply not set the corresponding
-- function pointers of the @GdkPixbufModule@ structure. If a module supports
-- incremental loading (i.e. provides @begin_load@, @stop_load@ and
-- @load_increment@), it doesn\'t have to implement @load@, since @GdkPixbuf@
-- can supply a generic @load@ implementation wrapping the incremental loading.
-- 
-- == Installing modules
-- 
-- Installing a module is a two-step process:
-- 
-- * copy the module file(s) to the loader directory (normally
--   @$libdir\/gdk-pixbuf-2.0\/$version\/loaders@, unless overridden by the
--   environment variable @GDK_PIXBUF_MODULEDIR@)
-- * call @gdk-pixbuf-query-loaders@ to update the module file (normally
--   @$libdir\/gdk-pixbuf-2.0\/$version\/loaders.cache@, unless overridden
--   by the environment variable @GDK_PIXBUF_MODULE_FILE@)
-- 

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.GdkPixbuf.Structs.PixbufModule
    ( 

-- * Exported types
    PixbufModule(..)                        ,
    newZeroPixbufModule                     ,


 -- * Methods

#if defined(ENABLE_OVERLOADING)
    ResolvePixbufModuleMethod               ,
#endif



 -- * Properties


-- ** info #attr:info#
-- | a @GdkPixbufFormat@ holding information about the module.

    clearPixbufModuleInfo                   ,
    getPixbufModuleInfo                     ,
#if defined(ENABLE_OVERLOADING)
    pixbufModule_info                       ,
#endif
    setPixbufModuleInfo                     ,


-- ** isSaveOptionSupported #attr:isSaveOptionSupported#
-- | returns whether a save option key is supported by the module

    clearPixbufModuleIsSaveOptionSupported  ,
    getPixbufModuleIsSaveOptionSupported    ,
#if defined(ENABLE_OVERLOADING)
    pixbufModule_isSaveOptionSupported      ,
#endif
    setPixbufModuleIsSaveOptionSupported    ,


-- ** load #attr:load#
-- | loads an image from a file.

    clearPixbufModuleLoad                   ,
    getPixbufModuleLoad                     ,
#if defined(ENABLE_OVERLOADING)
    pixbufModule_load                       ,
#endif
    setPixbufModuleLoad                     ,


-- ** loadAnimation #attr:loadAnimation#
-- | loads an animation from a file.

    clearPixbufModuleLoadAnimation          ,
    getPixbufModuleLoadAnimation            ,
#if defined(ENABLE_OVERLOADING)
    pixbufModule_loadAnimation              ,
#endif
    setPixbufModuleLoadAnimation            ,


-- ** loadIncrement #attr:loadIncrement#
-- | continues an incremental load.

    clearPixbufModuleLoadIncrement          ,
    getPixbufModuleLoadIncrement            ,
#if defined(ENABLE_OVERLOADING)
    pixbufModule_loadIncrement              ,
#endif
    setPixbufModuleLoadIncrement            ,


-- ** loadXpmData #attr:loadXpmData#
-- | loads an image from data in memory.

    clearPixbufModuleLoadXpmData            ,
    getPixbufModuleLoadXpmData              ,
#if defined(ENABLE_OVERLOADING)
    pixbufModule_loadXpmData                ,
#endif
    setPixbufModuleLoadXpmData              ,


-- ** module #attr:module#
-- | the loaded @GModule@.

    clearPixbufModuleModule                 ,
    getPixbufModuleModule                   ,
#if defined(ENABLE_OVERLOADING)
    pixbufModule_module                     ,
#endif
    setPixbufModuleModule                   ,


-- ** moduleName #attr:moduleName#
-- | the name of the module, usually the same as the
--  usual file extension for images of this type, eg. \"xpm\", \"jpeg\" or \"png\".

    clearPixbufModuleModuleName             ,
    getPixbufModuleModuleName               ,
#if defined(ENABLE_OVERLOADING)
    pixbufModule_moduleName                 ,
#endif
    setPixbufModuleModuleName               ,


-- ** modulePath #attr:modulePath#
-- | the path from which the module is loaded.

    clearPixbufModuleModulePath             ,
    getPixbufModuleModulePath               ,
#if defined(ENABLE_OVERLOADING)
    pixbufModule_modulePath                 ,
#endif
    setPixbufModuleModulePath               ,


-- ** save #attr:save#
-- | saves a @GdkPixbuf@ to a file.

    clearPixbufModuleSave                   ,
    getPixbufModuleSave                     ,
#if defined(ENABLE_OVERLOADING)
    pixbufModule_save                       ,
#endif
    setPixbufModuleSave                     ,


-- ** stopLoad #attr:stopLoad#
-- | stops an incremental load.

    clearPixbufModuleStopLoad               ,
    getPixbufModuleStopLoad                 ,
#if defined(ENABLE_OVERLOADING)
    pixbufModule_stopLoad                   ,
#endif
    setPixbufModuleStopLoad                 ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GModule.Structs.Module as GModule.Module
import qualified GI.GdkPixbuf.Callbacks as GdkPixbuf.Callbacks
import {-# SOURCE #-} qualified GI.GdkPixbuf.Structs.PixbufFormat as GdkPixbuf.PixbufFormat
import {-# SOURCE #-} qualified GI.GdkPixbuf.Structs.PixbufModulePattern as GdkPixbuf.PixbufModulePattern

#else
import qualified GI.GModule.Structs.Module as GModule.Module
import qualified GI.GdkPixbuf.Callbacks as GdkPixbuf.Callbacks
import {-# SOURCE #-} qualified GI.GdkPixbuf.Structs.PixbufFormat as GdkPixbuf.PixbufFormat

#endif

-- | Memory-managed wrapper type.
newtype PixbufModule = PixbufModule (SP.ManagedPtr PixbufModule)
    deriving (Eq)

instance SP.ManagedPtrNewtype PixbufModule where
    toManagedPtr (PixbufModule p) = p

instance BoxedPtr PixbufModule where
    boxedPtrCopy = \p -> B.ManagedPtr.withManagedPtr p (copyBytes 136 >=> B.ManagedPtr.wrapPtr PixbufModule)
    boxedPtrFree = \x -> SP.withManagedPtr x SP.freeMem
instance CallocPtr PixbufModule where
    boxedPtrCalloc = callocBytes 136


-- | Construct a t'PixbufModule' struct initialized to zero.
newZeroPixbufModule :: MonadIO m => m PixbufModule
newZeroPixbufModule = liftIO $ boxedPtrCalloc >>= wrapPtr PixbufModule

instance tag ~ 'AttrSet => Constructible PixbufModule tag where
    new _ attrs = do
        o <- newZeroPixbufModule
        GI.Attributes.set o attrs
        return o


-- | Get the value of the “@module_name@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' pixbufModule #moduleName
-- @
getPixbufModuleModuleName :: MonadIO m => PixbufModule -> m (Maybe T.Text)
getPixbufModuleModuleName s = liftIO $ withManagedPtr s $ \ptr -> do
    val <- peek (ptr `plusPtr` 0) :: IO CString
    result <- SP.convertIfNonNull val $ \val' -> do
        val'' <- cstringToText val'
        return val''
    return result

-- | Set the value of the “@module_name@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' pixbufModule [ #moduleName 'Data.GI.Base.Attributes.:=' value ]
-- @
setPixbufModuleModuleName :: MonadIO m => PixbufModule -> CString -> m ()
setPixbufModuleModuleName s val = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 0) (val :: CString)

-- | Set the value of the “@module_name@” field to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #moduleName
-- @
clearPixbufModuleModuleName :: MonadIO m => PixbufModule -> m ()
clearPixbufModuleModuleName s = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 0) (FP.nullPtr :: CString)

#if defined(ENABLE_OVERLOADING)
data PixbufModuleModuleNameFieldInfo
instance AttrInfo PixbufModuleModuleNameFieldInfo where
    type AttrBaseTypeConstraint PixbufModuleModuleNameFieldInfo = (~) PixbufModule
    type AttrAllowedOps PixbufModuleModuleNameFieldInfo = '[ 'AttrSet, 'AttrGet, 'AttrClear]
    type AttrSetTypeConstraint PixbufModuleModuleNameFieldInfo = (~) CString
    type AttrTransferTypeConstraint PixbufModuleModuleNameFieldInfo = (~)CString
    type AttrTransferType PixbufModuleModuleNameFieldInfo = CString
    type AttrGetType PixbufModuleModuleNameFieldInfo = Maybe T.Text
    type AttrLabel PixbufModuleModuleNameFieldInfo = "module_name"
    type AttrOrigin PixbufModuleModuleNameFieldInfo = PixbufModule
    attrGet = getPixbufModuleModuleName
    attrSet = setPixbufModuleModuleName
    attrConstruct = undefined
    attrClear = clearPixbufModuleModuleName
    attrTransfer _ v = do
        return v
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GdkPixbuf.Structs.PixbufModule.moduleName"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdkpixbuf-2.0.32/docs/GI-GdkPixbuf-Structs-PixbufModule.html#g:attr:moduleName"
        })

pixbufModule_moduleName :: AttrLabelProxy "moduleName"
pixbufModule_moduleName = AttrLabelProxy

#endif


-- | Get the value of the “@module_path@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' pixbufModule #modulePath
-- @
getPixbufModuleModulePath :: MonadIO m => PixbufModule -> m (Maybe T.Text)
getPixbufModuleModulePath s = liftIO $ withManagedPtr s $ \ptr -> do
    val <- peek (ptr `plusPtr` 8) :: IO CString
    result <- SP.convertIfNonNull val $ \val' -> do
        val'' <- cstringToText val'
        return val''
    return result

-- | Set the value of the “@module_path@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' pixbufModule [ #modulePath 'Data.GI.Base.Attributes.:=' value ]
-- @
setPixbufModuleModulePath :: MonadIO m => PixbufModule -> CString -> m ()
setPixbufModuleModulePath s val = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 8) (val :: CString)

-- | Set the value of the “@module_path@” field to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #modulePath
-- @
clearPixbufModuleModulePath :: MonadIO m => PixbufModule -> m ()
clearPixbufModuleModulePath s = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 8) (FP.nullPtr :: CString)

#if defined(ENABLE_OVERLOADING)
data PixbufModuleModulePathFieldInfo
instance AttrInfo PixbufModuleModulePathFieldInfo where
    type AttrBaseTypeConstraint PixbufModuleModulePathFieldInfo = (~) PixbufModule
    type AttrAllowedOps PixbufModuleModulePathFieldInfo = '[ 'AttrSet, 'AttrGet, 'AttrClear]
    type AttrSetTypeConstraint PixbufModuleModulePathFieldInfo = (~) CString
    type AttrTransferTypeConstraint PixbufModuleModulePathFieldInfo = (~)CString
    type AttrTransferType PixbufModuleModulePathFieldInfo = CString
    type AttrGetType PixbufModuleModulePathFieldInfo = Maybe T.Text
    type AttrLabel PixbufModuleModulePathFieldInfo = "module_path"
    type AttrOrigin PixbufModuleModulePathFieldInfo = PixbufModule
    attrGet = getPixbufModuleModulePath
    attrSet = setPixbufModuleModulePath
    attrConstruct = undefined
    attrClear = clearPixbufModuleModulePath
    attrTransfer _ v = do
        return v
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GdkPixbuf.Structs.PixbufModule.modulePath"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdkpixbuf-2.0.32/docs/GI-GdkPixbuf-Structs-PixbufModule.html#g:attr:modulePath"
        })

pixbufModule_modulePath :: AttrLabelProxy "modulePath"
pixbufModule_modulePath = AttrLabelProxy

#endif


-- | Get the value of the “@module@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' pixbufModule #module
-- @
getPixbufModuleModule :: MonadIO m => PixbufModule -> m (Maybe GModule.Module.Module)
getPixbufModuleModule s = liftIO $ withManagedPtr s $ \ptr -> do
    val <- peek (ptr `plusPtr` 16) :: IO (Ptr GModule.Module.Module)
    result <- SP.convertIfNonNull val $ \val' -> do
        val'' <- (newPtr GModule.Module.Module) val'
        return val''
    return result

-- | Set the value of the “@module@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' pixbufModule [ #module 'Data.GI.Base.Attributes.:=' value ]
-- @
setPixbufModuleModule :: MonadIO m => PixbufModule -> Ptr GModule.Module.Module -> m ()
setPixbufModuleModule s val = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 16) (val :: Ptr GModule.Module.Module)

-- | Set the value of the “@module@” field to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #module
-- @
clearPixbufModuleModule :: MonadIO m => PixbufModule -> m ()
clearPixbufModuleModule s = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 16) (FP.nullPtr :: Ptr GModule.Module.Module)

#if defined(ENABLE_OVERLOADING)
data PixbufModuleModuleFieldInfo
instance AttrInfo PixbufModuleModuleFieldInfo where
    type AttrBaseTypeConstraint PixbufModuleModuleFieldInfo = (~) PixbufModule
    type AttrAllowedOps PixbufModuleModuleFieldInfo = '[ 'AttrSet, 'AttrGet, 'AttrClear]
    type AttrSetTypeConstraint PixbufModuleModuleFieldInfo = (~) (Ptr GModule.Module.Module)
    type AttrTransferTypeConstraint PixbufModuleModuleFieldInfo = (~)(Ptr GModule.Module.Module)
    type AttrTransferType PixbufModuleModuleFieldInfo = (Ptr GModule.Module.Module)
    type AttrGetType PixbufModuleModuleFieldInfo = Maybe GModule.Module.Module
    type AttrLabel PixbufModuleModuleFieldInfo = "module"
    type AttrOrigin PixbufModuleModuleFieldInfo = PixbufModule
    attrGet = getPixbufModuleModule
    attrSet = setPixbufModuleModule
    attrConstruct = undefined
    attrClear = clearPixbufModuleModule
    attrTransfer _ v = do
        return v
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GdkPixbuf.Structs.PixbufModule.module"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdkpixbuf-2.0.32/docs/GI-GdkPixbuf-Structs-PixbufModule.html#g:attr:module"
        })

pixbufModule_module :: AttrLabelProxy "module"
pixbufModule_module = AttrLabelProxy

#endif


-- | Get the value of the “@info@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' pixbufModule #info
-- @
getPixbufModuleInfo :: MonadIO m => PixbufModule -> m (Maybe GdkPixbuf.PixbufFormat.PixbufFormat)
getPixbufModuleInfo s = liftIO $ withManagedPtr s $ \ptr -> do
    val <- peek (ptr `plusPtr` 24) :: IO (Ptr GdkPixbuf.PixbufFormat.PixbufFormat)
    result <- SP.convertIfNonNull val $ \val' -> do
        val'' <- (newBoxed GdkPixbuf.PixbufFormat.PixbufFormat) val'
        return val''
    return result

-- | Set the value of the “@info@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' pixbufModule [ #info 'Data.GI.Base.Attributes.:=' value ]
-- @
setPixbufModuleInfo :: MonadIO m => PixbufModule -> Ptr GdkPixbuf.PixbufFormat.PixbufFormat -> m ()
setPixbufModuleInfo s val = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 24) (val :: Ptr GdkPixbuf.PixbufFormat.PixbufFormat)

-- | Set the value of the “@info@” field to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #info
-- @
clearPixbufModuleInfo :: MonadIO m => PixbufModule -> m ()
clearPixbufModuleInfo s = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 24) (FP.nullPtr :: Ptr GdkPixbuf.PixbufFormat.PixbufFormat)

#if defined(ENABLE_OVERLOADING)
data PixbufModuleInfoFieldInfo
instance AttrInfo PixbufModuleInfoFieldInfo where
    type AttrBaseTypeConstraint PixbufModuleInfoFieldInfo = (~) PixbufModule
    type AttrAllowedOps PixbufModuleInfoFieldInfo = '[ 'AttrSet, 'AttrGet, 'AttrClear]
    type AttrSetTypeConstraint PixbufModuleInfoFieldInfo = (~) (Ptr GdkPixbuf.PixbufFormat.PixbufFormat)
    type AttrTransferTypeConstraint PixbufModuleInfoFieldInfo = (~)(Ptr GdkPixbuf.PixbufFormat.PixbufFormat)
    type AttrTransferType PixbufModuleInfoFieldInfo = (Ptr GdkPixbuf.PixbufFormat.PixbufFormat)
    type AttrGetType PixbufModuleInfoFieldInfo = Maybe GdkPixbuf.PixbufFormat.PixbufFormat
    type AttrLabel PixbufModuleInfoFieldInfo = "info"
    type AttrOrigin PixbufModuleInfoFieldInfo = PixbufModule
    attrGet = getPixbufModuleInfo
    attrSet = setPixbufModuleInfo
    attrConstruct = undefined
    attrClear = clearPixbufModuleInfo
    attrTransfer _ v = do
        return v
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GdkPixbuf.Structs.PixbufModule.info"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdkpixbuf-2.0.32/docs/GI-GdkPixbuf-Structs-PixbufModule.html#g:attr:info"
        })

pixbufModule_info :: AttrLabelProxy "info"
pixbufModule_info = AttrLabelProxy

#endif


-- | Get the value of the “@load@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' pixbufModule #load
-- @
getPixbufModuleLoad :: MonadIO m => PixbufModule -> m (Maybe GdkPixbuf.Callbacks.PixbufModuleLoadFunc)
getPixbufModuleLoad s = liftIO $ withManagedPtr s $ \ptr -> do
    val <- peek (ptr `plusPtr` 32) :: IO (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadFunc)
    result <- SP.convertFunPtrIfNonNull val $ \val' -> do
        let val'' = GdkPixbuf.Callbacks.dynamic_PixbufModuleLoadFunc val'
        return val''
    return result

-- | Set the value of the “@load@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' pixbufModule [ #load 'Data.GI.Base.Attributes.:=' value ]
-- @
setPixbufModuleLoad :: MonadIO m => PixbufModule -> FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadFunc -> m ()
setPixbufModuleLoad s val = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 32) (val :: FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadFunc)

-- | Set the value of the “@load@” field to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #load
-- @
clearPixbufModuleLoad :: MonadIO m => PixbufModule -> m ()
clearPixbufModuleLoad s = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 32) (FP.nullFunPtr :: FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadFunc)

#if defined(ENABLE_OVERLOADING)
data PixbufModuleLoadFieldInfo
instance AttrInfo PixbufModuleLoadFieldInfo where
    type AttrBaseTypeConstraint PixbufModuleLoadFieldInfo = (~) PixbufModule
    type AttrAllowedOps PixbufModuleLoadFieldInfo = '[ 'AttrSet, 'AttrGet, 'AttrClear]
    type AttrSetTypeConstraint PixbufModuleLoadFieldInfo = (~) (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadFunc)
    type AttrTransferTypeConstraint PixbufModuleLoadFieldInfo = (~)(FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadFunc)
    type AttrTransferType PixbufModuleLoadFieldInfo = (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadFunc)
    type AttrGetType PixbufModuleLoadFieldInfo = Maybe GdkPixbuf.Callbacks.PixbufModuleLoadFunc
    type AttrLabel PixbufModuleLoadFieldInfo = "load"
    type AttrOrigin PixbufModuleLoadFieldInfo = PixbufModule
    attrGet = getPixbufModuleLoad
    attrSet = setPixbufModuleLoad
    attrConstruct = undefined
    attrClear = clearPixbufModuleLoad
    attrTransfer _ v = do
        return v
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GdkPixbuf.Structs.PixbufModule.load"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdkpixbuf-2.0.32/docs/GI-GdkPixbuf-Structs-PixbufModule.html#g:attr:load"
        })

pixbufModule_load :: AttrLabelProxy "load"
pixbufModule_load = AttrLabelProxy

#endif


-- | Get the value of the “@load_xpm_data@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' pixbufModule #loadXpmData
-- @
getPixbufModuleLoadXpmData :: MonadIO m => PixbufModule -> m (Maybe GdkPixbuf.Callbacks.PixbufModuleLoadXpmDataFunc)
getPixbufModuleLoadXpmData s = liftIO $ withManagedPtr s $ \ptr -> do
    val <- peek (ptr `plusPtr` 40) :: IO (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadXpmDataFunc)
    result <- SP.convertFunPtrIfNonNull val $ \val' -> do
        let val'' = GdkPixbuf.Callbacks.dynamic_PixbufModuleLoadXpmDataFunc val'
        return val''
    return result

-- | Set the value of the “@load_xpm_data@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' pixbufModule [ #loadXpmData 'Data.GI.Base.Attributes.:=' value ]
-- @
setPixbufModuleLoadXpmData :: MonadIO m => PixbufModule -> FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadXpmDataFunc -> m ()
setPixbufModuleLoadXpmData s val = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 40) (val :: FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadXpmDataFunc)

-- | Set the value of the “@load_xpm_data@” field to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #loadXpmData
-- @
clearPixbufModuleLoadXpmData :: MonadIO m => PixbufModule -> m ()
clearPixbufModuleLoadXpmData s = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 40) (FP.nullFunPtr :: FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadXpmDataFunc)

#if defined(ENABLE_OVERLOADING)
data PixbufModuleLoadXpmDataFieldInfo
instance AttrInfo PixbufModuleLoadXpmDataFieldInfo where
    type AttrBaseTypeConstraint PixbufModuleLoadXpmDataFieldInfo = (~) PixbufModule
    type AttrAllowedOps PixbufModuleLoadXpmDataFieldInfo = '[ 'AttrSet, 'AttrGet, 'AttrClear]
    type AttrSetTypeConstraint PixbufModuleLoadXpmDataFieldInfo = (~) (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadXpmDataFunc)
    type AttrTransferTypeConstraint PixbufModuleLoadXpmDataFieldInfo = (~)GdkPixbuf.Callbacks.PixbufModuleLoadXpmDataFunc
    type AttrTransferType PixbufModuleLoadXpmDataFieldInfo = (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadXpmDataFunc)
    type AttrGetType PixbufModuleLoadXpmDataFieldInfo = Maybe GdkPixbuf.Callbacks.PixbufModuleLoadXpmDataFunc
    type AttrLabel PixbufModuleLoadXpmDataFieldInfo = "load_xpm_data"
    type AttrOrigin PixbufModuleLoadXpmDataFieldInfo = PixbufModule
    attrGet = getPixbufModuleLoadXpmData
    attrSet = setPixbufModuleLoadXpmData
    attrConstruct = undefined
    attrClear = clearPixbufModuleLoadXpmData
    attrTransfer _ v = do
        GdkPixbuf.Callbacks.mk_PixbufModuleLoadXpmDataFunc (GdkPixbuf.Callbacks.wrap_PixbufModuleLoadXpmDataFunc Nothing v)
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GdkPixbuf.Structs.PixbufModule.loadXpmData"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdkpixbuf-2.0.32/docs/GI-GdkPixbuf-Structs-PixbufModule.html#g:attr:loadXpmData"
        })

pixbufModule_loadXpmData :: AttrLabelProxy "loadXpmData"
pixbufModule_loadXpmData = AttrLabelProxy

#endif


-- | Get the value of the “@stop_load@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' pixbufModule #stopLoad
-- @
getPixbufModuleStopLoad :: MonadIO m => PixbufModule -> m (Maybe GdkPixbuf.Callbacks.PixbufModuleStopLoadFunc)
getPixbufModuleStopLoad s = liftIO $ withManagedPtr s $ \ptr -> do
    val <- peek (ptr `plusPtr` 56) :: IO (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleStopLoadFunc)
    result <- SP.convertFunPtrIfNonNull val $ \val' -> do
        let val'' = GdkPixbuf.Callbacks.dynamic_PixbufModuleStopLoadFunc val'
        return val''
    return result

-- | Set the value of the “@stop_load@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' pixbufModule [ #stopLoad 'Data.GI.Base.Attributes.:=' value ]
-- @
setPixbufModuleStopLoad :: MonadIO m => PixbufModule -> FunPtr GdkPixbuf.Callbacks.C_PixbufModuleStopLoadFunc -> m ()
setPixbufModuleStopLoad s val = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 56) (val :: FunPtr GdkPixbuf.Callbacks.C_PixbufModuleStopLoadFunc)

-- | Set the value of the “@stop_load@” field to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #stopLoad
-- @
clearPixbufModuleStopLoad :: MonadIO m => PixbufModule -> m ()
clearPixbufModuleStopLoad s = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 56) (FP.nullFunPtr :: FunPtr GdkPixbuf.Callbacks.C_PixbufModuleStopLoadFunc)

#if defined(ENABLE_OVERLOADING)
data PixbufModuleStopLoadFieldInfo
instance AttrInfo PixbufModuleStopLoadFieldInfo where
    type AttrBaseTypeConstraint PixbufModuleStopLoadFieldInfo = (~) PixbufModule
    type AttrAllowedOps PixbufModuleStopLoadFieldInfo = '[ 'AttrSet, 'AttrGet, 'AttrClear]
    type AttrSetTypeConstraint PixbufModuleStopLoadFieldInfo = (~) (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleStopLoadFunc)
    type AttrTransferTypeConstraint PixbufModuleStopLoadFieldInfo = (~)(FunPtr GdkPixbuf.Callbacks.C_PixbufModuleStopLoadFunc)
    type AttrTransferType PixbufModuleStopLoadFieldInfo = (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleStopLoadFunc)
    type AttrGetType PixbufModuleStopLoadFieldInfo = Maybe GdkPixbuf.Callbacks.PixbufModuleStopLoadFunc
    type AttrLabel PixbufModuleStopLoadFieldInfo = "stop_load"
    type AttrOrigin PixbufModuleStopLoadFieldInfo = PixbufModule
    attrGet = getPixbufModuleStopLoad
    attrSet = setPixbufModuleStopLoad
    attrConstruct = undefined
    attrClear = clearPixbufModuleStopLoad
    attrTransfer _ v = do
        return v
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GdkPixbuf.Structs.PixbufModule.stopLoad"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdkpixbuf-2.0.32/docs/GI-GdkPixbuf-Structs-PixbufModule.html#g:attr:stopLoad"
        })

pixbufModule_stopLoad :: AttrLabelProxy "stopLoad"
pixbufModule_stopLoad = AttrLabelProxy

#endif


-- | Get the value of the “@load_increment@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' pixbufModule #loadIncrement
-- @
getPixbufModuleLoadIncrement :: MonadIO m => PixbufModule -> m (Maybe GdkPixbuf.Callbacks.PixbufModuleIncrementLoadFunc)
getPixbufModuleLoadIncrement s = liftIO $ withManagedPtr s $ \ptr -> do
    val <- peek (ptr `plusPtr` 64) :: IO (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleIncrementLoadFunc)
    result <- SP.convertFunPtrIfNonNull val $ \val' -> do
        let val'' = GdkPixbuf.Callbacks.dynamic_PixbufModuleIncrementLoadFunc val'
        return val''
    return result

-- | Set the value of the “@load_increment@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' pixbufModule [ #loadIncrement 'Data.GI.Base.Attributes.:=' value ]
-- @
setPixbufModuleLoadIncrement :: MonadIO m => PixbufModule -> FunPtr GdkPixbuf.Callbacks.C_PixbufModuleIncrementLoadFunc -> m ()
setPixbufModuleLoadIncrement s val = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 64) (val :: FunPtr GdkPixbuf.Callbacks.C_PixbufModuleIncrementLoadFunc)

-- | Set the value of the “@load_increment@” field to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #loadIncrement
-- @
clearPixbufModuleLoadIncrement :: MonadIO m => PixbufModule -> m ()
clearPixbufModuleLoadIncrement s = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 64) (FP.nullFunPtr :: FunPtr GdkPixbuf.Callbacks.C_PixbufModuleIncrementLoadFunc)

#if defined(ENABLE_OVERLOADING)
data PixbufModuleLoadIncrementFieldInfo
instance AttrInfo PixbufModuleLoadIncrementFieldInfo where
    type AttrBaseTypeConstraint PixbufModuleLoadIncrementFieldInfo = (~) PixbufModule
    type AttrAllowedOps PixbufModuleLoadIncrementFieldInfo = '[ 'AttrSet, 'AttrGet, 'AttrClear]
    type AttrSetTypeConstraint PixbufModuleLoadIncrementFieldInfo = (~) (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleIncrementLoadFunc)
    type AttrTransferTypeConstraint PixbufModuleLoadIncrementFieldInfo = (~)(FunPtr GdkPixbuf.Callbacks.C_PixbufModuleIncrementLoadFunc)
    type AttrTransferType PixbufModuleLoadIncrementFieldInfo = (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleIncrementLoadFunc)
    type AttrGetType PixbufModuleLoadIncrementFieldInfo = Maybe GdkPixbuf.Callbacks.PixbufModuleIncrementLoadFunc
    type AttrLabel PixbufModuleLoadIncrementFieldInfo = "load_increment"
    type AttrOrigin PixbufModuleLoadIncrementFieldInfo = PixbufModule
    attrGet = getPixbufModuleLoadIncrement
    attrSet = setPixbufModuleLoadIncrement
    attrConstruct = undefined
    attrClear = clearPixbufModuleLoadIncrement
    attrTransfer _ v = do
        return v
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GdkPixbuf.Structs.PixbufModule.loadIncrement"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdkpixbuf-2.0.32/docs/GI-GdkPixbuf-Structs-PixbufModule.html#g:attr:loadIncrement"
        })

pixbufModule_loadIncrement :: AttrLabelProxy "loadIncrement"
pixbufModule_loadIncrement = AttrLabelProxy

#endif


-- | Get the value of the “@load_animation@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' pixbufModule #loadAnimation
-- @
getPixbufModuleLoadAnimation :: MonadIO m => PixbufModule -> m (Maybe GdkPixbuf.Callbacks.PixbufModuleLoadAnimationFunc)
getPixbufModuleLoadAnimation s = liftIO $ withManagedPtr s $ \ptr -> do
    val <- peek (ptr `plusPtr` 72) :: IO (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadAnimationFunc)
    result <- SP.convertFunPtrIfNonNull val $ \val' -> do
        let val'' = GdkPixbuf.Callbacks.dynamic_PixbufModuleLoadAnimationFunc val'
        return val''
    return result

-- | Set the value of the “@load_animation@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' pixbufModule [ #loadAnimation 'Data.GI.Base.Attributes.:=' value ]
-- @
setPixbufModuleLoadAnimation :: MonadIO m => PixbufModule -> FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadAnimationFunc -> m ()
setPixbufModuleLoadAnimation s val = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 72) (val :: FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadAnimationFunc)

-- | Set the value of the “@load_animation@” field to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #loadAnimation
-- @
clearPixbufModuleLoadAnimation :: MonadIO m => PixbufModule -> m ()
clearPixbufModuleLoadAnimation s = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 72) (FP.nullFunPtr :: FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadAnimationFunc)

#if defined(ENABLE_OVERLOADING)
data PixbufModuleLoadAnimationFieldInfo
instance AttrInfo PixbufModuleLoadAnimationFieldInfo where
    type AttrBaseTypeConstraint PixbufModuleLoadAnimationFieldInfo = (~) PixbufModule
    type AttrAllowedOps PixbufModuleLoadAnimationFieldInfo = '[ 'AttrSet, 'AttrGet, 'AttrClear]
    type AttrSetTypeConstraint PixbufModuleLoadAnimationFieldInfo = (~) (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadAnimationFunc)
    type AttrTransferTypeConstraint PixbufModuleLoadAnimationFieldInfo = (~)(FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadAnimationFunc)
    type AttrTransferType PixbufModuleLoadAnimationFieldInfo = (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleLoadAnimationFunc)
    type AttrGetType PixbufModuleLoadAnimationFieldInfo = Maybe GdkPixbuf.Callbacks.PixbufModuleLoadAnimationFunc
    type AttrLabel PixbufModuleLoadAnimationFieldInfo = "load_animation"
    type AttrOrigin PixbufModuleLoadAnimationFieldInfo = PixbufModule
    attrGet = getPixbufModuleLoadAnimation
    attrSet = setPixbufModuleLoadAnimation
    attrConstruct = undefined
    attrClear = clearPixbufModuleLoadAnimation
    attrTransfer _ v = do
        return v
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GdkPixbuf.Structs.PixbufModule.loadAnimation"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdkpixbuf-2.0.32/docs/GI-GdkPixbuf-Structs-PixbufModule.html#g:attr:loadAnimation"
        })

pixbufModule_loadAnimation :: AttrLabelProxy "loadAnimation"
pixbufModule_loadAnimation = AttrLabelProxy

#endif


-- | Get the value of the “@save@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' pixbufModule #save
-- @
getPixbufModuleSave :: MonadIO m => PixbufModule -> m (Maybe GdkPixbuf.Callbacks.PixbufModuleSaveFunc)
getPixbufModuleSave s = liftIO $ withManagedPtr s $ \ptr -> do
    val <- peek (ptr `plusPtr` 80) :: IO (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleSaveFunc)
    result <- SP.convertFunPtrIfNonNull val $ \val' -> do
        let val'' = GdkPixbuf.Callbacks.dynamic_PixbufModuleSaveFunc val'
        return val''
    return result

-- | Set the value of the “@save@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' pixbufModule [ #save 'Data.GI.Base.Attributes.:=' value ]
-- @
setPixbufModuleSave :: MonadIO m => PixbufModule -> FunPtr GdkPixbuf.Callbacks.C_PixbufModuleSaveFunc -> m ()
setPixbufModuleSave s val = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 80) (val :: FunPtr GdkPixbuf.Callbacks.C_PixbufModuleSaveFunc)

-- | Set the value of the “@save@” field to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #save
-- @
clearPixbufModuleSave :: MonadIO m => PixbufModule -> m ()
clearPixbufModuleSave s = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 80) (FP.nullFunPtr :: FunPtr GdkPixbuf.Callbacks.C_PixbufModuleSaveFunc)

#if defined(ENABLE_OVERLOADING)
data PixbufModuleSaveFieldInfo
instance AttrInfo PixbufModuleSaveFieldInfo where
    type AttrBaseTypeConstraint PixbufModuleSaveFieldInfo = (~) PixbufModule
    type AttrAllowedOps PixbufModuleSaveFieldInfo = '[ 'AttrSet, 'AttrGet, 'AttrClear]
    type AttrSetTypeConstraint PixbufModuleSaveFieldInfo = (~) (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleSaveFunc)
    type AttrTransferTypeConstraint PixbufModuleSaveFieldInfo = (~)(FunPtr GdkPixbuf.Callbacks.C_PixbufModuleSaveFunc)
    type AttrTransferType PixbufModuleSaveFieldInfo = (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleSaveFunc)
    type AttrGetType PixbufModuleSaveFieldInfo = Maybe GdkPixbuf.Callbacks.PixbufModuleSaveFunc
    type AttrLabel PixbufModuleSaveFieldInfo = "save"
    type AttrOrigin PixbufModuleSaveFieldInfo = PixbufModule
    attrGet = getPixbufModuleSave
    attrSet = setPixbufModuleSave
    attrConstruct = undefined
    attrClear = clearPixbufModuleSave
    attrTransfer _ v = do
        return v
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GdkPixbuf.Structs.PixbufModule.save"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdkpixbuf-2.0.32/docs/GI-GdkPixbuf-Structs-PixbufModule.html#g:attr:save"
        })

pixbufModule_save :: AttrLabelProxy "save"
pixbufModule_save = AttrLabelProxy

#endif


-- | Get the value of the “@is_save_option_supported@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' pixbufModule #isSaveOptionSupported
-- @
getPixbufModuleIsSaveOptionSupported :: MonadIO m => PixbufModule -> m (Maybe GdkPixbuf.Callbacks.PixbufModuleSaveOptionSupportedFunc)
getPixbufModuleIsSaveOptionSupported s = liftIO $ withManagedPtr s $ \ptr -> do
    val <- peek (ptr `plusPtr` 96) :: IO (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleSaveOptionSupportedFunc)
    result <- SP.convertFunPtrIfNonNull val $ \val' -> do
        let val'' = GdkPixbuf.Callbacks.dynamic_PixbufModuleSaveOptionSupportedFunc val'
        return val''
    return result

-- | Set the value of the “@is_save_option_supported@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' pixbufModule [ #isSaveOptionSupported 'Data.GI.Base.Attributes.:=' value ]
-- @
setPixbufModuleIsSaveOptionSupported :: MonadIO m => PixbufModule -> FunPtr GdkPixbuf.Callbacks.C_PixbufModuleSaveOptionSupportedFunc -> m ()
setPixbufModuleIsSaveOptionSupported s val = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 96) (val :: FunPtr GdkPixbuf.Callbacks.C_PixbufModuleSaveOptionSupportedFunc)

-- | Set the value of the “@is_save_option_supported@” field to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #isSaveOptionSupported
-- @
clearPixbufModuleIsSaveOptionSupported :: MonadIO m => PixbufModule -> m ()
clearPixbufModuleIsSaveOptionSupported s = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 96) (FP.nullFunPtr :: FunPtr GdkPixbuf.Callbacks.C_PixbufModuleSaveOptionSupportedFunc)

#if defined(ENABLE_OVERLOADING)
data PixbufModuleIsSaveOptionSupportedFieldInfo
instance AttrInfo PixbufModuleIsSaveOptionSupportedFieldInfo where
    type AttrBaseTypeConstraint PixbufModuleIsSaveOptionSupportedFieldInfo = (~) PixbufModule
    type AttrAllowedOps PixbufModuleIsSaveOptionSupportedFieldInfo = '[ 'AttrSet, 'AttrGet, 'AttrClear]
    type AttrSetTypeConstraint PixbufModuleIsSaveOptionSupportedFieldInfo = (~) (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleSaveOptionSupportedFunc)
    type AttrTransferTypeConstraint PixbufModuleIsSaveOptionSupportedFieldInfo = (~)GdkPixbuf.Callbacks.PixbufModuleSaveOptionSupportedFunc
    type AttrTransferType PixbufModuleIsSaveOptionSupportedFieldInfo = (FunPtr GdkPixbuf.Callbacks.C_PixbufModuleSaveOptionSupportedFunc)
    type AttrGetType PixbufModuleIsSaveOptionSupportedFieldInfo = Maybe GdkPixbuf.Callbacks.PixbufModuleSaveOptionSupportedFunc
    type AttrLabel PixbufModuleIsSaveOptionSupportedFieldInfo = "is_save_option_supported"
    type AttrOrigin PixbufModuleIsSaveOptionSupportedFieldInfo = PixbufModule
    attrGet = getPixbufModuleIsSaveOptionSupported
    attrSet = setPixbufModuleIsSaveOptionSupported
    attrConstruct = undefined
    attrClear = clearPixbufModuleIsSaveOptionSupported
    attrTransfer _ v = do
        GdkPixbuf.Callbacks.mk_PixbufModuleSaveOptionSupportedFunc (GdkPixbuf.Callbacks.wrap_PixbufModuleSaveOptionSupportedFunc Nothing v)
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GdkPixbuf.Structs.PixbufModule.isSaveOptionSupported"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gdkpixbuf-2.0.32/docs/GI-GdkPixbuf-Structs-PixbufModule.html#g:attr:isSaveOptionSupported"
        })

pixbufModule_isSaveOptionSupported :: AttrLabelProxy "isSaveOptionSupported"
pixbufModule_isSaveOptionSupported = AttrLabelProxy

#endif



#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList PixbufModule
type instance O.AttributeList PixbufModule = PixbufModuleAttributeList
type PixbufModuleAttributeList = ('[ '("moduleName", PixbufModuleModuleNameFieldInfo), '("modulePath", PixbufModuleModulePathFieldInfo), '("module", PixbufModuleModuleFieldInfo), '("info", PixbufModuleInfoFieldInfo), '("load", PixbufModuleLoadFieldInfo), '("loadXpmData", PixbufModuleLoadXpmDataFieldInfo), '("stopLoad", PixbufModuleStopLoadFieldInfo), '("loadIncrement", PixbufModuleLoadIncrementFieldInfo), '("loadAnimation", PixbufModuleLoadAnimationFieldInfo), '("save", PixbufModuleSaveFieldInfo), '("isSaveOptionSupported", PixbufModuleIsSaveOptionSupportedFieldInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolvePixbufModuleMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolvePixbufModuleMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolvePixbufModuleMethod t PixbufModule, O.OverloadedMethod info PixbufModule p) => OL.IsLabel t (PixbufModule -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolvePixbufModuleMethod t PixbufModule, O.OverloadedMethod info PixbufModule p, R.HasField t PixbufModule p) => R.HasField t PixbufModule p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolvePixbufModuleMethod t PixbufModule, O.OverloadedMethodInfo info PixbufModule) => OL.IsLabel t (O.MethodProxy info PixbufModule) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


