﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/wisdom/ConnectWisdomServiceRequest.h>
#include <aws/wisdom/ConnectWisdomService_EXPORTS.h>
#include <aws/wisdom/model/KnowledgeBaseType.h>
#include <aws/wisdom/model/RenderingConfiguration.h>
#include <aws/wisdom/model/ServerSideEncryptionConfiguration.h>
#include <aws/wisdom/model/SourceConfiguration.h>

#include <utility>

namespace Aws {
namespace ConnectWisdomService {
namespace Model {

/**
 */
class CreateKnowledgeBaseRequest : public ConnectWisdomServiceRequest {
 public:
  AWS_CONNECTWISDOMSERVICE_API CreateKnowledgeBaseRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateKnowledgeBase"; }

  AWS_CONNECTWISDOMSERVICE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. If not provided, the Amazon Web Services SDK
   * populates this field. For more information about idempotency, see <a
   * href="https://aws.amazon.com/builders-library/making-retries-safe-with-idempotent-APIs/">Making
   * retries safe with idempotent APIs</a>.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateKnowledgeBaseRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateKnowledgeBaseRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of knowledge base. Only CUSTOM knowledge bases allow you to upload
   * your own content. EXTERNAL knowledge bases support integrations with third-party
   * systems whose content is synchronized automatically. </p>
   */
  inline KnowledgeBaseType GetKnowledgeBaseType() const { return m_knowledgeBaseType; }
  inline bool KnowledgeBaseTypeHasBeenSet() const { return m_knowledgeBaseTypeHasBeenSet; }
  inline void SetKnowledgeBaseType(KnowledgeBaseType value) {
    m_knowledgeBaseTypeHasBeenSet = true;
    m_knowledgeBaseType = value;
  }
  inline CreateKnowledgeBaseRequest& WithKnowledgeBaseType(KnowledgeBaseType value) {
    SetKnowledgeBaseType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the knowledge base.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateKnowledgeBaseRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about how to render the content.</p>
   */
  inline const RenderingConfiguration& GetRenderingConfiguration() const { return m_renderingConfiguration; }
  inline bool RenderingConfigurationHasBeenSet() const { return m_renderingConfigurationHasBeenSet; }
  template <typename RenderingConfigurationT = RenderingConfiguration>
  void SetRenderingConfiguration(RenderingConfigurationT&& value) {
    m_renderingConfigurationHasBeenSet = true;
    m_renderingConfiguration = std::forward<RenderingConfigurationT>(value);
  }
  template <typename RenderingConfigurationT = RenderingConfiguration>
  CreateKnowledgeBaseRequest& WithRenderingConfiguration(RenderingConfigurationT&& value) {
    SetRenderingConfiguration(std::forward<RenderingConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration information for the customer managed key used for
   * encryption. </p> <p>This KMS key must have a policy that allows
   * <code>kms:CreateGrant</code>, <code>kms:DescribeKey</code>, and
   * <code>kms:Decrypt/kms:GenerateDataKey</code> permissions to the IAM identity
   * using the key to invoke Wisdom.</p> <p>For more information about setting up a
   * customer managed key for Wisdom, see <a
   * href="https://docs.aws.amazon.com/connect/latest/adminguide/enable-wisdom.html">Enable
   * Amazon Connect Wisdom for your instance</a>.</p>
   */
  inline const ServerSideEncryptionConfiguration& GetServerSideEncryptionConfiguration() const {
    return m_serverSideEncryptionConfiguration;
  }
  inline bool ServerSideEncryptionConfigurationHasBeenSet() const { return m_serverSideEncryptionConfigurationHasBeenSet; }
  template <typename ServerSideEncryptionConfigurationT = ServerSideEncryptionConfiguration>
  void SetServerSideEncryptionConfiguration(ServerSideEncryptionConfigurationT&& value) {
    m_serverSideEncryptionConfigurationHasBeenSet = true;
    m_serverSideEncryptionConfiguration = std::forward<ServerSideEncryptionConfigurationT>(value);
  }
  template <typename ServerSideEncryptionConfigurationT = ServerSideEncryptionConfiguration>
  CreateKnowledgeBaseRequest& WithServerSideEncryptionConfiguration(ServerSideEncryptionConfigurationT&& value) {
    SetServerSideEncryptionConfiguration(std::forward<ServerSideEncryptionConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The source of the knowledge base content. Only set this argument for EXTERNAL
   * knowledge bases.</p>
   */
  inline const SourceConfiguration& GetSourceConfiguration() const { return m_sourceConfiguration; }
  inline bool SourceConfigurationHasBeenSet() const { return m_sourceConfigurationHasBeenSet; }
  template <typename SourceConfigurationT = SourceConfiguration>
  void SetSourceConfiguration(SourceConfigurationT&& value) {
    m_sourceConfigurationHasBeenSet = true;
    m_sourceConfiguration = std::forward<SourceConfigurationT>(value);
  }
  template <typename SourceConfigurationT = SourceConfiguration>
  CreateKnowledgeBaseRequest& WithSourceConfiguration(SourceConfigurationT&& value) {
    SetSourceConfiguration(std::forward<SourceConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags used to organize, track, or control access for this resource.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateKnowledgeBaseRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateKnowledgeBaseRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  KnowledgeBaseType m_knowledgeBaseType{KnowledgeBaseType::NOT_SET};
  bool m_knowledgeBaseTypeHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  RenderingConfiguration m_renderingConfiguration;
  bool m_renderingConfigurationHasBeenSet = false;

  ServerSideEncryptionConfiguration m_serverSideEncryptionConfiguration;
  bool m_serverSideEncryptionConfigurationHasBeenSet = false;

  SourceConfiguration m_sourceConfiguration;
  bool m_sourceConfigurationHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ConnectWisdomService
}  // namespace Aws
